// WiseMindAI Web Clipper popup
// 保存链接到 WiseMindAI

const DEFAULT_HOST_PORT = '127.0.0.1:38221';
const API_PATH = '/api/create-file';

const titleInput = document.getElementById('titleInput');
const urlInput = document.getElementById('urlInput');
const iconPreview = document.getElementById('iconPreview');
const iconPlaceholder = document.getElementById('iconPlaceholder');
const saveBtn = document.getElementById('saveBtn');
const openSettingBtn = document.getElementById('openSettingBtn');
const statusEl = document.getElementById('status');
const helpTextEl = document.getElementById('helpText');

function setStatus(message, type = '') {
  statusEl.textContent = message || '';
  statusEl.className = 'status';
  if (type) {
    statusEl.classList.add(type);
  }
  helpTextEl.style.display = 'none';
  helpTextEl.textContent = '';
}

function showHelpText(message, isHtml = false) {
  if (isHtml) {
    helpTextEl.innerHTML = message;
  } else {
    helpTextEl.textContent = message;
  }
  helpTextEl.style.display = 'block';
}

function loadHostPort() {
  return new Promise(resolve => {
    if (!chrome.storage || !chrome.storage.sync) {
      resolve(DEFAULT_HOST_PORT);
      return;
    }
    // 优先从新配置读取，兼容旧配置
    chrome.storage.sync.get(['wisemindConfig', 'wisemindHostPort'], result => {
      const config = result.wisemindConfig || {};
      resolve(config.apiEndpoint || result.wisemindHostPort || DEFAULT_HOST_PORT);
    });
  });
}

function buildEndpoint(hostPort) {
  if (!hostPort) return null;
  let raw = hostPort.trim();
  if (!raw) return null;

  if (!raw.startsWith('http://') && !raw.startsWith('https://')) {
    raw = `http://${raw}`;
  }

  try {
    const url = new URL(raw);
    const port = url.port || (raw.startsWith('https://') ? '443' : '80');
    const hostPortStr = `${url.hostname}:${port}`;
    return `http://${hostPortStr}${API_PATH}`;
  } catch {
    return null;
  }
}

// 检查是否是支持的 URL 协议
function isSupportedUrl(url) {
  if (!url) return false;
  const supportedSchemes = ['http://', 'https://', 'file://'];
  return supportedSchemes.some(scheme => url.startsWith(scheme));
}

async function init() {
  if (!titleInput || !urlInput || !iconPreview || !iconPlaceholder) {
    console.error('DOM elements not found');
    return;
  }

  if (!chrome.tabs) {
    setStatus('✗ Chrome Tabs API 不可用', 'error');
    return;
  }

  try {
    // 从 URL 参数中获取标签页 ID（如果是从悬浮球打开的）
    const urlParams = new URLSearchParams(window.location.search);
    const tabIdParam = urlParams.get('tabId');

    let tab = null;

    if (tabIdParam) {
      // 使用传递的标签页 ID
      try {
        tab = await chrome.tabs.get(parseInt(tabIdParam, 10));
      } catch (e) {
        console.warn('Failed to get tab by ID:', e);
      }
    }

    // 如果没有从参数获取到，尝试从当前窗口获取
    if (!tab) {
      const tabs = await chrome.tabs.query({ active: true, currentWindow: true });
      tab = tabs && tabs[0];
    }

    // 如果还是没有，尝试获取所有窗口中的活动标签页
    if (!tab) {
      const tabs = await chrome.tabs.query({ active: true });
      tab = tabs && tabs[0];
    }

    if (!tab) {
      setStatus('未找到活动标签页', 'error');
      return;
    }

    // 检查是否是支持的 URL
    if (!isSupportedUrl(tab.url)) {
      setStatus('✗ 系统页面不支持保存', 'error');
      if (saveBtn) {
        saveBtn.disabled = true;
        saveBtn.style.opacity = '0.5';
        saveBtn.style.cursor = 'not-allowed';
      }
      return;
    }

    if (titleInput) {
      titleInput.value = tab.title || '';
    }
    if (urlInput) {
      urlInput.value = tab.url || '';
    }

    if (tab.favIconUrl && iconPreview && iconPlaceholder) {
      iconPreview.src = tab.favIconUrl;
      iconPreview.style.display = 'block';
      iconPlaceholder.style.display = 'none';
    } else if (iconPreview && iconPlaceholder) {
      iconPreview.style.display = 'none';
      iconPlaceholder.style.display = 'flex';
    }
  } catch (error) {
    console.error('Failed to get tab info:', error);
    setStatus('✗ 获取页面信息失败', 'error');
  }
}

async function saveToWiseMind() {
  setStatus('');
  const hostPort = await loadHostPort();
  const endpoint = buildEndpoint(hostPort);

  if (!endpoint) {
    setStatus('✗ API 端点配置错误，请先配置正确的域名和端口', 'error');
    return;
  }

  const url = urlInput.value.trim();
  const title = titleInput.value.trim() || url;
  const icon = iconPreview.style.display === 'block' ? iconPreview.src : '';

  if (!url) {
    setStatus('✗ 链接不能为空', 'error');
    return;
  }

  // 获取网页纯文本内容
  let pageContent = '';
  try {
    const tabs = await chrome.tabs.query({ active: true, currentWindow: true });
    const tab = tabs && tabs[0];
    if (tab && tab.id) {
      const response = await chrome.tabs.sendMessage(tab.id, { type: 'WISEMIND_GET_PAGE_TEXT' });
      if (response && response.ok && response.text) {
        pageContent = response.text;
      }
    }
  } catch (e) {
    // 某些页面无法注入 content script，忽略错误
    console.warn('Failed to get page text content:', e);
  }

  const payload = {
    filePath: url,
    name: title,
    size: 0,
    type: 'link',
    fileType: 'link',
    icon: icon,
    content: pageContent,
  };

  try {
    setStatus('正在保存...', 'info');
    const res = await fetch(endpoint, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
      },
      body: JSON.stringify(payload),
    });

    if (!res.ok) {
      const text = await res.text().catch(() => '');
      throw new Error(text || `HTTP ${res.status}`);
    }

    await res.json();
    setStatus('✓ 已成功保存到 WiseMindAI', 'success');
  } catch (err) {
    const errorMsg = err?.message || String(err);
    const isConnectionError =
      errorMsg.includes('Failed to fetch') ||
      errorMsg.includes('NetworkError') ||
      errorMsg.includes('ERR_CONNECTION_REFUSED') ||
      errorMsg.includes('ERR_INTERNET_DISCONNECTED');

    if (isConnectionError) {
      setStatus('✗ 连接失败', 'error');
      showHelpText(
        '无法连接到 WiseMindAI 服务。请确保：<ol style="margin-top:4px;padding-left:18px;">' +
        '<li>WiseMindAI 客户端已启动</li>' +
        '<li>在 WiseMindAI 设置中已启用"本地 API 服务"</li>' +
        '<li>端口号与配置一致（默认：38221）</li>' +
        '</ol>',
        true,
      );
    } else {
      setStatus(`✗ 保存失败: ${errorMsg}`, 'error');
    }
  }
}

saveBtn?.addEventListener('click', () => {
  saveToWiseMind();
});

openSettingBtn?.addEventListener('click', () => {
  chrome.runtime.openOptionsPage();
});

if (document.readyState === 'loading') {
  document.addEventListener('DOMContentLoaded', () => {
    init().catch(err => {
      console.error('Init error:', err);
      setStatus(`✗ 初始化失败: ${err?.message || err}`, 'error');
    });
  });
} else {
  init().catch(err => {
    console.error('Init error:', err);
    setStatus(`✗ 初始化失败: ${err?.message || err}`, 'error');
  });
}
